﻿# Native PKCE quick test for a xSale Rest Api Client .

$AUTH0_CLIENT_ID = " *** UZUPEŁNIJ *** "
$ORGANIZATION_NAME =  " *** UZUPEŁNIJ *** "

$REDIRECT_URI = "http://localhost:8400/callback"

$AUTH0_DOMAIN = "login.futuriti.pl"
$AUTH0_AUDIENCE = "https://api.xsale.ai"
$SCOPE = "openid profile email xsale production offline_access"
$XSALE_API_BASE_URL = "https://api.xsale.ai"

# Optional: only when Auth0 Organizations is enabled.
$AUTH0_ORGANIZATION = ""

function Assert-ConfiguredValue {
  param(
    [Parameter(Mandatory = $true)][string]$Name,
    [Parameter(Mandatory = $true)][string]$Value
  )

  if ([string]::IsNullOrWhiteSpace($Value) -or $Value -like "your-*") {
    throw "Set proper value for $Name before running this script."
  }
}

function ConvertTo-Base64Url {
  param([byte[]]$Bytes)
  [Convert]::ToBase64String($Bytes).TrimEnd('=') -replace '\+', '-' -replace '/', '_'
}

Assert-ConfiguredValue -Name "AUTH0_DOMAIN" -Value $AUTH0_DOMAIN
Assert-ConfiguredValue -Name "AUTH0_CLIENT_ID" -Value $AUTH0_CLIENT_ID
Assert-ConfiguredValue -Name "AUTH0_AUDIENCE" -Value $AUTH0_AUDIENCE
Assert-ConfiguredValue -Name "ORGANIZATION_NAME" -Value $ORGANIZATION_NAME

# PKCE verifier/challenge
$random = New-Object byte[] 64
[System.Security.Cryptography.RandomNumberGenerator]::Create().GetBytes($random)
$codeVerifier = ConvertTo-Base64Url -Bytes $random

$sha = [System.Security.Cryptography.SHA256]::Create()
$challengeBytes = $sha.ComputeHash([System.Text.Encoding]::ASCII.GetBytes($codeVerifier))
$codeChallenge = ConvertTo-Base64Url -Bytes $challengeBytes

$queryParams = [ordered]@{
  response_type = "code"
  client_id = $AUTH0_CLIENT_ID
  redirect_uri = $REDIRECT_URI
  scope = $SCOPE
  audience = $AUTH0_AUDIENCE
  code_challenge = $codeChallenge
  code_challenge_method = "S256"
}

if (-not [string]::IsNullOrWhiteSpace($AUTH0_ORGANIZATION)) {
  $queryParams["organization"] = $AUTH0_ORGANIZATION
}

$queryString = ($queryParams.GetEnumerator() | ForEach-Object {
  "$([Uri]::EscapeDataString([string]$_.Key))=$([Uri]::EscapeDataString([string]$_.Value))"
}) -join "&"

$authorizeUrl = "https://$AUTH0_DOMAIN/authorize?$queryString"

Write-Host "1) Open URL in browser and log in:"
Write-Host $authorizeUrl
Write-Host ""
Write-Host "2) After redirect, copy 'code' from callback URL."
$authCode = Read-Host "Paste authorization code"
if ([string]::IsNullOrWhiteSpace($authCode)) {
  throw "Authorization code is required."
}

$tokenBody = @{
  grant_type = "authorization_code"
  client_id = $AUTH0_CLIENT_ID
  code_verifier = $codeVerifier
  code = $authCode
  redirect_uri = $REDIRECT_URI
} | ConvertTo-Json

$tokenResponse = Invoke-RestMethod -Method Post `
  -Uri "https://$AUTH0_DOMAIN/oauth/token" `
  -ContentType "application/json" `
  -Body $tokenBody

if (-not $tokenResponse.access_token) {
  throw "No access_token in token response."
}
if ([string]::IsNullOrWhiteSpace($tokenResponse.id_token)) {
  throw "No id_token in token response. x-id-token is required by xSale API."
}

Write-Host "Token acquired. Expires in: $($tokenResponse.expires_in) seconds"
Write-Host "id_token acquired (x-id-token is required by xSale API)."

$headers = @{ Authorization = "Bearer $($tokenResponse.access_token)" }
if (-not [string]::IsNullOrWhiteSpace($tokenResponse.id_token)) {
  $headers["x-id-token"] = $tokenResponse.id_token
}
$apiBase = $XSALE_API_BASE_URL.TrimEnd('/')

function Show-JsonPreview {
  param(
    [Parameter(Mandatory = $true)]
    $Value,
    [int]$MaxChars = 500
  )

  $json = $Value | ConvertTo-Json -Depth 10 -Compress
  if ($json.Length -gt $MaxChars) {
    return $json.Substring(0, $MaxChars) + "..."
  }

  return $json
}

function Get-AuthHeaders {
  param(
    $TokenResponse,
    [string]$FallbackIdToken = ""
  )

  $authHeaders = @{ Authorization = "Bearer $($TokenResponse.access_token)" }
  $idTokenToUse = $TokenResponse.id_token
  if ([string]::IsNullOrWhiteSpace($idTokenToUse)) {
    $idTokenToUse = $FallbackIdToken
  }
  if ([string]::IsNullOrWhiteSpace($idTokenToUse)) {
    throw "No id_token available. x-id-token is required by xSale API."
  }
  $authHeaders["x-id-token"] = $idTokenToUse
  return $authHeaders
}

$meResponse = Invoke-RestMethod -Method Get `
  -Uri "$apiBase/me" `
  -Headers $headers

Write-Host "GET /me -> response JSON (start):"
Write-Host (Show-JsonPreview -Value $meResponse)

$articlesResponse = Invoke-RestMethod -Method Get `
  -Uri "$apiBase/$ORGANIZATION_NAME/articles" `
  -Headers $headers

Write-Host "GET /$ORGANIZATION_NAME/articles -> response JSON (start):"
Write-Host (Show-JsonPreview -Value $articlesResponse)

if (-not [string]::IsNullOrWhiteSpace($tokenResponse.refresh_token)) {
  Write-Host ""
  Write-Host "Refreshing token via refresh_token grant..."

  $refreshBody = @{
    grant_type = "refresh_token"
    client_id = $AUTH0_CLIENT_ID
    refresh_token = $tokenResponse.refresh_token
    scope = $SCOPE
    audience = $AUTH0_AUDIENCE
  } | ConvertTo-Json

  $refreshResponse = Invoke-RestMethod -Method Post `
    -Uri "https://$AUTH0_DOMAIN/oauth/token" `
    -ContentType "application/json" `
    -Body $refreshBody

  if (-not $refreshResponse.access_token) {
    throw "No access_token in refresh response."
  }

  Write-Host "Refresh OK. New token expires in: $($refreshResponse.expires_in) seconds"
  $headers = Get-AuthHeaders -TokenResponse $refreshResponse -FallbackIdToken $tokenResponse.id_token

  $meAfterRefresh = Invoke-RestMethod -Method Get `
    -Uri "$apiBase/me" `
    -Headers $headers

  Write-Host "GET /me after refresh -> response JSON (start):"
  Write-Host (Show-JsonPreview -Value $meAfterRefresh)
} else {
  Write-Host ""
  Write-Host "No refresh_token in token response (check Auth0 app settings for offline access / refresh token rotation)."
}
